import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Sparkles, Star, Calendar, Building2, Award, Heart, TrendingUp, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyVisits.css'

const MyVisits = () => {
  const [visits, setVisits] = useState([])

  useEffect(() => {
    loadVisits()
    const interval = setInterval(loadVisits, 500)
    return () => clearInterval(interval)
  }, [])

  const loadVisits = () => {
    const saved = localStorage.getItem('spaVisits')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setVisits(parsed)
      } catch (e) {
        setVisits([])
      }
    } else {
      const defaultVisits = [
        {
          id: 1,
          procedure: 'Массаж',
          salon: 'SpaLux Premium',
          date: '2025-03-15',
          rating: 5
        },
        {
          id: 2,
          procedure: 'Пилинг',
          salon: 'SpaLux Center',
          date: '2025-03-10',
          rating: 4
        },
        {
          id: 3,
          procedure: 'Обёртывание',
          salon: 'SpaLux Premium',
          date: '2025-03-05',
          rating: 5
        }
      ]
      setVisits(defaultVisits)
      localStorage.setItem('spaVisits', JSON.stringify(defaultVisits))
    }
  }

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={16}
        fill={i < rating ? '#F59E0B' : 'none'}
        stroke={i < rating ? '#F59E0B' : 'var(--text-light)'}
        strokeWidth={1.5}
      />
    ))
  }

  const totalVisits = visits.length
  const averageRating = visits.length > 0 
    ? (visits.reduce((sum, v) => sum + v.rating, 0) / visits.length).toFixed(1)
    : 0
  const uniqueSalons = new Set(visits.map(v => v.salon)).size

  return (
    <div className="my-visits">
      <div className="container">
        <div className="page-header">
          <motion.h1 
            className="page-title"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6 }}
          >
            Мои посещения
          </motion.h1>
          <motion.p 
            className="page-subtitle"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            История ваших спа-процедур
          </motion.p>
        </div>

        <motion.div 
          className="stats-minimal"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{totalVisits}</div>
            <div className="stat-minimal-label">Посещений</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{averageRating}</div>
            <div className="stat-minimal-label">Рейтинг</div>
          </div>
          <div className="stat-minimal-item">
            <div className="stat-minimal-value">{uniqueSalons}</div>
            <div className="stat-minimal-label">Салона</div>
          </div>
        </motion.div>

        <ImageSlider images={[
          '/images/spa-1.jpg',
          '/images/spa-2.jpg',
          '/images/spa-3.jpg',
          '/images/spa-4.jpg'
        ]} />

        <motion.section 
          className="visits-list"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title-minimal">Ваши процедуры</h2>
          {visits.length > 0 ? (
            <div className="visits-vertical">
              {visits.map((visit, index) => (
                <motion.div 
                  key={visit.id} 
                  className="visit-item"
                  initial={{ opacity: 0, x: -30 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ x: 8, borderColor: 'var(--primary)' }}
                >
                  <div className="visit-item-left">
                    <div className="visit-item-icon">
                      <Sparkles size={24} strokeWidth={2} />
                    </div>
                    <div className="visit-item-content">
                      <div className="visit-item-header">
                        <h3 className="visit-item-procedure">{visit.procedure}</h3>
                        <div className="visit-item-rating">
                          {renderStars(visit.rating)}
                        </div>
                      </div>
                      <div className="visit-item-details">
                        <div className="visit-item-detail">
                          <Building2 size={16} strokeWidth={2} />
                          <span>{visit.salon}</span>
                        </div>
                        <div className="visit-item-detail">
                          <Calendar size={16} strokeWidth={2} />
                          <span>
                            {new Date(visit.date).toLocaleDateString('ru-RU', { 
                              day: 'numeric', 
                              month: 'long', 
                              year: 'numeric' 
                            })}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <Sparkles size={64} strokeWidth={1.5} className="empty-icon" />
              <h3 className="empty-title">Нет посещений</h3>
              <p className="empty-text">Запишитесь на процедуру</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="features-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <h2 className="section-title-minimal">Преимущества</h2>
          <div className="features-grid">
            <div className="feature-item">
              <div className="feature-icon">
                <Award size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Премиум-качество</h3>
                <p className="feature-text">Лучшие процедуры и средства</p>
              </div>
            </div>
            <div className="feature-item">
              <div className="feature-icon">
                <Heart size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Забота о здоровье</h3>
                <p className="feature-text">Натуральные компоненты</p>
              </div>
            </div>
            <div className="feature-item">
              <div className="feature-icon">
                <Clock size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Гибкий график</h3>
                <p className="feature-text">Запись в удобное время</p>
              </div>
            </div>
            <div className="feature-item">
              <div className="feature-icon">
                <TrendingUp size={24} strokeWidth={2} />
              </div>
              <div className="feature-content">
                <h3 className="feature-title">Высокий рейтинг</h3>
                <p className="feature-text">99% довольных клиентов</p>
              </div>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MyVisits

